#include "InmalyzeWIFI.h"

InmalyzeWIFI::InmalyzeWIFI() : server(80) {}

void InmalyzeWIFI::begin(const char* ssid, const char* password) {
  // Start Serial only if not already started, but we assume user might start it
  // We only print the IP as requested.
  
  WiFi.begin(ssid, password);

  // Wait for connection (Silent loop)
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
  }

  // ONLY print the IP, nothing else
  Serial.println(WiFi.localIP());

  // Setup Routes
  server.on("/", HTTP_GET, std::bind(&InmalyzeWIFI::handleRoot, this));
  server.on("/control", HTTP_GET, std::bind(&InmalyzeWIFI::handleControl, this));
  server.on("/", HTTP_OPTIONS, std::bind(&InmalyzeWIFI::handleOptions, this));
  server.on("/control", HTTP_OPTIONS, std::bind(&InmalyzeWIFI::handleOptions, this));

  server.begin();
}

void InmalyzeWIFI::loop() {
  server.handleClient();
}

// ===== Data Handling =====

void InmalyzeWIFI::send(String key, String value) {
  dataStore[key] = value;
}

void InmalyzeWIFI::send(String key, int value) {
  dataStore[key] = String(value);
}

void InmalyzeWIFI::send(String key, float value) {
  dataStore[key] = String(value, 1); // Default to 1 decimal place
}

String InmalyzeWIFI::get(String key) {
  if (dataStore.find(key) != dataStore.end()) {
    return dataStore[key];
  }
  return ""; // Return empty if key doesn't exist
}

// ===== Server Handlers =====

void InmalyzeWIFI::setCORSHeaders() {
  server.sendHeader("Access-Control-Allow-Origin", "*");
  server.sendHeader("Access-Control-Allow-Methods", "GET, POST, OPTIONS");
  server.sendHeader("Access-Control-Allow-Headers", "Content-Type");
}

void InmalyzeWIFI::handleRoot() {
  setCORSHeaders();

  // specific manual JSON construction to avoid external dependencies like ArduinoJson
  String json = "{";
  int count = 0;
  for (auto const& [key, val] : dataStore) {
    if (count > 0) json += ",";
    
    // Check if value looks like a number, if so don't use quotes, else use quotes
    // For simplicity in this specific request, we will treat values as raw strings/numbers
    // If the value contains only digits or a dot, send as number, else string.
    bool isNumber = true;
    for (char c : val) {
        if (!isdigit(c) && c != '.' && c != '-') {
            isNumber = false;
            break;
        }
    }

    json += "\"" + key + "\":";
    if (isNumber && val.length() > 0) {
        json += val;
    } else {
        json += "\"" + val + "\"";
    }
    count++;
  }
  json += "}";

  server.send(200, "application/json", json);
}

void InmalyzeWIFI::handleControl() {
  setCORSHeaders();

  if (server.hasArg("var") && server.hasArg("val")) {
    String variable = server.arg("var");
    String value = server.arg("val");

    // Update the local storage
    dataStore[variable] = value;

    server.send(200, "text/plain", "OK");
  } else {
    server.send(400, "text/plain", "Missing parameters");
  }
}

void InmalyzeWIFI::handleOptions() {
  setCORSHeaders();
  server.send(200, "text/plain", "");
}